<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Models\Coupon;
use App\Models\Product;
use Gloudemans\Shoppingcart\Facades\Cart;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Auth;
use App\Helpers\Supports;

class CartController extends Controller
{

    //Mini Cart Start

    //Product Add To Mini Cart
    public function addToCard(Request $request, $id)
    {
    
        $request->validate([
            'quantity'=>'required',
            'price'=>'required',
            'unit'=>'required',
            ]);
            
        if (Session::has('coupon')) {
            Session::forget('coupon');
        }
        $product     = Product::findOrFail($id);
        $color       = $request->color;
        $size        = $request->size;
        $quantity    = $request->quantity ?? 1;
        $productName = $product->product_name_en;
        $price = intval($request->price);
        $unit = $product->unit;
        
        if ($product->discount == null) {
            Cart::add([
                'id'      => $id,
                'name'    => $product->product_name_en,
                'qty'     => $quantity,
                'weight'  => $request->weight,
                'price'   => $price,
                'options' => [
                    'size'  => $size,
                    'color' => $color,
                    'image' => $product->product_thumbnail,
                    'unit' => $unit,
                    'thickness' => $request->thickness
                ],
            ]);
            return response()->json($productName);
        }
        else {
            Cart::add([
                'id'      => $id,
                'name'    => $product->product_name_en,
                'qty'     => $quantity,
                'weight'  => $request->weight,
                'price'   => $price,
                'options' => [
                    'size'  => $size,
                    'color' => $color,
                    'image' => $product->product_thumbnail,
                    'unit' => $unit,
                ],
            ]);
            return response()->json($productName);
        }

    }
    //Product Add To Mini Cart
    public function addCard(Request $request, $id)
    { 
  
         
        if (Session::has('coupon')) {
            Session::forget('coupon');
        }
        
        $product     = Product::findOrFail($id);
        $color       = $request->color;
        $size        = $request->size;
        $quantity    = $request->quantity ?? 1;
        $productName = $product->product_name_en;
        $price = intval($request->price);
        $unit = $product->unit;

        $cart = Cart::add([
            'id'      => $id,
            'name'    => $product->product_name_en,
            'qty'     => $quantity,
            'weight'  => 0,
            'price'   => $price,
            'options' => [
                'size'  => $size,
                'color' => $color,
                'image' => $product->product_thumbnail,
                'unit' => $unit,
                'is_checkout' => true,
                'thickness' => $request->thickness
            ],
        ]);
        
        return redirect()->back()->withSuccess('Product Added to Cart');
        
        
    }
    //View Mini Cart Products
    public function viewCart()
    {
        $products = Cart::content();
        $qty      = Cart::content()->count();
        $total    = Cart::total();
        return response()->json([
            'products' => $products,
            'qty'      => $qty,
            'total'    => $total,
        ]);
    }

    //Remove Product From Mini Cart
    public function removeFromCart($rowId)
    {

        Cart::remove($rowId);
        
        if (Session::has('coupon')) {
            Session::forget('coupon');
        }
        
        return redirect()->back()->withSuccess('Product Remove From Your Cart');
        

    }
    public function remove($rowId)
    {

        Cart::remove($rowId);
        
        if (Session::has('coupon')) {
            Session::forget('coupon');
        }
        
        return redirect()->back()->withSuccess('Product Remove From Your Cart');
        
    }
    //Mini Cart End

    //Main Cart start
    public function index()
    {

        $brands = Brand::orderBy('brand_name_en', 'ASC')->get();
        $carts = Cart::content();
        
        $checkoutAll = $carts->every(function ($item) {
            return $item->options->has('is_checkout') && $item->options->get('is_checkout') === true;
        });
        
        return view('user.cart', compact('brands','carts','checkoutAll'));

    }

    //Apply Coupon
    public function applyCoupon(Request $request)
    {
        $couponCode = $request->coupon_code;
        
        
        $coupon     = Coupon::where('coupon_code', $couponCode)->first();
        

        if (!$coupon){
            return response()->json(['error' => 'Your Coupon is Invalid. Please Apply a valid coupon.']);
        }
      
        
        if ($coupon->email && !Auth::check()){
            return response()->json(['error' => 'Please Login First to get this coupon discount.']);
        }
        
        
        if ($coupon->email && Auth::user()->email != $coupon->email){
            return response()->json(['error' => 'Your Coupon is Invalid. Please Apply a valid coupon.']);
        }
        $cartTotalAmount = Supports::getCartTotal();
        
        if($cartTotalAmount <= 0){
            return response()->json(['error' => 'Your Checkout Total is 0. Please Select item for checkout.']);
        }
        
        
        if ($coupon->validity >= Carbon::now()->format('Y-m-d')) {
            $amount         = Supports::getCartTotal();
            $discount       = $coupon->discount;
            $discountAmount = ($amount * $discount) / 100;
            $totalAmount    = $amount - $discountAmount;
            Session::put('coupon', [
                'couponCode'     => $couponCode,
                'couponDiscount' => $coupon->discount,
                'discount'       => $discountAmount,
                'subtotal'       => $amount,
                'totalAmount'    => $totalAmount,
            ]);
            
            $message  = Auth::check() ? 
                "{$couponCode} Applied Successfully. You got {$discount}% Special Discount  From Steel Bazar." : 
                "{$couponCode} Applied Successfully and You got {$discount}% discount.";
            
            return response()->json(['success' => $message]);
        }
        else{
            return response()->json(['error' => 'Your Coupon is Expired']);
        }
        
    }
    //Get Coupon Discount
    public function couponDiscount()
    {
        if (Session::has('coupon'))
        {
            $subtotal = round(Supports::getCartTotal());
            $coupon   = session()->get('coupon')['couponCode'];
            $discount = session()->get('coupon')['discount'];
            $total    = session()->get('coupon')['totalAmount'];
            return response()->json([
                'subtotal'    => $subtotal,
                'coupon'      => $coupon,
                'discount'    => $discount,
                'totalAmount' => $total,
                'hasCoupon' => true,
            ]);
        }
        else
        {
            $total = Supports::getCartTotal();
            return response()->json([
                'cartTotal' => $total,
                'hasCoupon' => false,
            ]);
        }
    }
    //Remove Coupon
    public function removeCoupon()
    {

        Session::forget('coupon');
        return response()->json(['success' => 'Coupon successfully Removed']);

    }

    //Increase Qty
    public function increaseQty($rawId)
    {
        $product = Cart::get($rawId);
        $updatedQty = $product->qty + 1;
        Cart::update($rawId, $updatedQty);
        if (Session::has('coupon'))
        {
            $amount         = Supports::getCartTotal();
            $couponCode     = session()->get('coupon')['couponCode'];
            $couponDiscount = session()->get('coupon')['couponDiscount'];
            $discountAmount = ($amount * $couponDiscount) / 100;
            $totalAmount    = $amount - $discountAmount;
            Session::put('coupon', [
                'couponCode'     => $couponCode,
                'couponDiscount' => $couponDiscount,
                'discount'       => $discountAmount,
                'subtotal'       => $amount,
                'totalAmount'    => $totalAmount,
            ]);

        }
         return response()->json(['status'=>'success','qty'=>$updatedQty]);
    }

    //Decrease Qty
    public function decreaseQty($rawId)
    {
        $product = Cart::get($rawId);
        if ($product->qty > 1)
        {
            
            $updatedQty = $product->qty - 1;
            Cart::update($rawId, $updatedQty);
            
            if (Session::has('coupon'))
            {
                $amount         = round(Supports::getCartTotal());
                $couponCode     = session()->get('coupon')['couponCode'];
                $couponDiscount = session()->get('coupon')['couponDiscount'];
                $discountAmount = ($amount * $couponDiscount) / 100;
                $totalAmount    = $amount - $discountAmount;
                Session::put('coupon', [
                    'couponCode'     => $couponCode,
                    'couponDiscount' => $couponDiscount,
                    'discount'       => $discountAmount,
                    'subtotal'       => $amount,
                    'totalAmount'    => $totalAmount,
                ]);

            }
            return response()->json(['status'=>'success','qty'=>$updatedQty]);
        }

    }
    
    public function checkoutAction(Request $request)
    {
        $cartItems = Cart::content();
        
        $checkoutAll = $cartItems->every(function ($item) {
            return $item->options->has('is_checkout') && $item->options->get('is_checkout') === true;
        });
        
        if($request->is_checkout){
            foreach($cartItems as $item){
                $proudct = Cart::get($item->rowId);
                $options = [
                    
                    'size'  => $proudct->options['size'],
                    'color' => $proudct->options['color'],
                    'image' => $proudct->options['image'],
                    'unit' => $proudct->options['unit'],
                    'is_checkout' => true
                ];
                Cart::update($item->rowId, ['options'  => $options]);
            }
            
        }else {
            foreach($cartItems as $item){
                $proudct = Cart::get($item->rowId);
                $options = [
                    
                    'size'  => $proudct->options['size'],
                    'color' => $proudct->options['color'],
                    'image' => $proudct->options['image'],
                    'unit' => $proudct->options['unit'],
                    'is_checkout' => false
                ];
                Cart::update($item->rowId, ['options'  => $options]);
            }
        
        }

        
        return response()->json(['status'=>'success']);
    }
    public function singleCheckoutAction($rowId)
    {
       
        $proudct = Cart::get($rowId);
        $options = [
                    
            'size'  => $proudct->options['size'],
            'color' => $proudct->options['color'],
            'image' => $proudct->options['image'],
            'unit' => $proudct->options['unit'],
            'is_checkout' => !$proudct->options['is_checkout']
        ];
        Cart::update($rowId, ['options'  => $options]);
        return response()->json(['status'=>'success']);
    }

}