<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Attribute;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Product;
use App\Models\ProductAttribute;
use App\Models\ProductMultipleImage;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Intervention\Image\Facades\Image;

class ProductController extends Controller
{
    //Show Products
    public function index()
    {
        $products = Product::with('images')->latest()->paginate(10);
        
        return view('admin.product.index', compact('products'));
    }
    //Add Product
    public function create()
    {
        $brands     = Brand::orderBy('brand_name_en', 'ASC')->get();
        $categories = Category::orderBy('category_name_en', 'ASC')->get();
        $attributes = Attribute::orderBy('priyority', 'ASC')->get();
        return view('admin.product.create', compact('brands', 'categories', 'attributes'));
    }
    //Store Product
    public function store(Request $request)
    {
 
        $request->validate([
            'brand_id'               => 'nullable | integer',
            'category_id'            => 'required | integer',
            'subcategory_id'         => 'nullable | integer',
            'subsubcategory_id'      => 'nullable | integer',
            'product_name_en'        => 'required | string',
            'product_name_bn'        => 'nullable | string',
            'product_code'           => 'required | string',
            'product_quantity_en'    => 'required | string',
            'product_quantity_bn'    => 'nullable | string',
            'product_tags_en'        => 'required | string',
            'product_tags_bn'        => 'nullable | string',
            'product_color_en'       => 'required | string',
            'product_color_bn'       => 'nullable | string',
            'product_size_en'        => 'nullable | string',
            'product_size_bn'        => 'nullable | string',
            'product_weight_en'      => 'nullable | string',
            'product_weight_bn'      => 'nullable | string',
            'unit'                   => 'required | string',
            'selling_price'          => 'required | string',
            'discount'               => 'nullable | integer',
            'discount_type'          => 'nullable | string',
            'product_thumbnail'      => 'required',
            'product_multiple_image' => 'required',
            'hightlight_attribute'   => 'nullable',
            'product_type'           => 'nullable',
            'hot_deals'              => 'nullable',
            'featured'               => 'nullable',
            'special_offer'          => 'nullable',
            'special_deals'          => 'nullable',
            'short_desc_en'          => 'required',
            'short_desc_bn'          => 'nullable',
            'full_desc_en'           => 'required',
            'full_desc_bn'           => 'nullable',
            'is_price_variant'       => 'boolean',
            'hide_size_weight'       => 'boolean',
            'variants'               => 'nullable | array',
        ], [
            'brand_id.required'               => 'Please Select A Brand',
            'category_id.required'            => 'Please Select A Category',
            'subcategory_id.required'         => 'Please Select A Sub Category',
            'subsubcategory_id.required'      => 'Please Select A Sub Sub Category',
            'product_name_en.required'        => 'Please Enter Product Name English',
            'product_code.required'           => 'Please Enter Product Code',
            'product_quantity.required'       => 'Please Enter Product Quantity',
            'product_tags_en.required'        => 'Please Enter Product Tags English',
            'product_color_en.required'       => 'Please Enter Product Color English',
            'product_color_bn.required'       => 'Please Enter Product Color Bangla',
            'product_size_en.required'        => 'Please Enter Product Size English',
            'selling_price.required'          => 'Please Enter Product Selling Price',
            'product_thumbnail.required'      => 'Please Enter Product Thumbnail',
            'product_multiple_image.required' => 'Please Enter Product Images',
            'short_desc_en.required'          => 'Enter Product Short Description English',
            'full_desc_en.required'           => 'Enter Product Full Description English',
        ]);
        
        if($request->is_price_variant){
            $request->validate([
                'variants.*.unit' => 'required',
                'variants.*.price' => 'required',
                'variants.*.convert_value' => 'required',
            ]);
        }
        if ($request->discount_type = 'parcent' && intval($request->discount) > 99)
        {
            return redirect()->back()->withError("Discount Should be less than 100");
        }
        try {
            
            if ($request->hasFile('product_thumbnail')) {
                $productThumbnail = $request->file('product_thumbnail');
                $thumbnailName    = uniqid() . time() . '.' . $productThumbnail->extension();
                Image::make($productThumbnail)->resize(917, 1000)->save('uploads/products/thumbnails/' . $thumbnailName);
                $thumbnailUrl = 'uploads/products/thumbnails/' . $thumbnailName;
            }
            if ($request->discount_type === 'parcent') {
                $discountAmount = ($request->selling_price * $request->discount) / 100;
                $discountPrice  = round($request->selling_price - $discountAmount);
            }
            else {
                $discountPrice = round($request->selling_price - $request->discount);
            }
            $product = Product::create([
                'brand_id'             => $request->brand_id,
                'category_id'          => $request->category_id,
                'subcategory_id'       => $request->subcategory_id,
                'subsubcategory_id'    => $request->subsubcategory_id,
                'product_name_en'      => $request->product_name_en,
                'product_name_bn'      => $request->product_name_bn,
                'product_slug_en'      => Str::slug($request->product_name_en, '-'),
                'product_code'         => $request->product_code,
                'product_quantity_en'  => $request->product_quantity_en,
                'product_quantity_bn'  => $request->product_quantity_bn,
                'product_tags_en'      => $request->product_tags_en,
                'product_tags_bn'      => $request->product_tags_bn,
                'product_color_en'     => $request->product_color_en,
                'product_color_bn'     => $request->product_color_bn,
                'product_size_en'      => $request->product_size_en,
                'product_size_bn'      => $request->product_size_bn,
                'product_weight_en'    => $request->product_weight_en,
                'product_weight_bn'    => $request->product_weight_bn,
                'unit'                 => $request->unit,
                'selling_price'        => $request->selling_price,
                'discount'             => $request->discount,
                'discount_type'        => $request->discount_type,
                'discount_price'       => $discountPrice,
                'is_price_variant'     => $request->is_price_variant,
                'price_variants'       => $request->variants,
                'product_thumbnail'    => $thumbnailUrl,
                'thumbnail_size'       => round($productThumbnail->getSize()/1024,2),
                'hightlight_attribute' => $request->hightlight_attribute,
                'product_type'         => $request->product_type,
                'hot_deals'            => $request->hot_deals,
                'featured'             => $request->featured,
                'special_offer'        => $request->special_offer,
                'special_deals'        => $request->special_deals,
                'hide_size_weight'     => $request->hide_size_weight,
                'short_desc_en'        => $request->short_desc_en,
                'short_desc_bn'        => $request->short_desc_bn,
                'full_desc_en'         => $request->full_desc_en,
                'full_desc_bn'         => $request->full_desc_bn,
            ]);
            if ($request->hasFile('product_multiple_image'))
            {
                $productImages = $request->file('product_multiple_image');
                foreach ($productImages as $productImage)
                {
                    $imageName = uniqid() . time() . '.' . $productImage->extension();
                    Image::make($productImage)->resize(917, 1000)->save('uploads/products/multiple-images/' . $imageName);
                    $imageUrl = 'uploads/products/multiple-images/' . $imageName;
                    $product->images()->create([
                        'product_image_name' => $imageUrl,
                        'product_image_size' => round($productImage->getSize()/1024,2),
                    ]);
                }
            }

            foreach ($request->product_attributes as $key => $product_attribute)
            {

                $product->productAttributes()->create([
                    'attribute_name'  => $key,
                    'attribute_value' => $product_attribute,
                ]);
            }
            return redirect()->route('products.all')->withSuccess("{$request->product_name_en} Added Sccessfully");
        }
        catch (\Exception$e)
        {
            return redirect()->back()->withInput()->withError($e->getMessage());
        }
    }
    public function edit($productId)
    {
        $product    = Product::with('productAttributes', 'category', 'subCategory', 'subSubCategory')->findOrFail($productId);
        $brands     = Brand::orderBy('brand_name_en', 'ASC')->get();
        $categories = Category::orderBy('category_name_en', 'ASC')->get();
        $attributes = Attribute::orderBy('priyority', 'ASC')->get();

        return view('admin.product.edit', compact('product', 'brands', 'categories', 'attributes'));
    }
    public function update(Request $request, $productId)
    {

        $request->validate([
            'brand_id'             => 'nullable | integer',
            'category_id'          => 'required | integer',
            'subcategory_id'       => 'nullable | integer',
            'subsubcategory_id'    => 'nullable | integer',
            'product_name_en'      => 'required | string',
            'product_name_bn'      => 'nullable | string',
            'product_code'         => 'required | string',
            'product_quantity_en'  => 'required | string',
            'product_quantity_bn'  => 'nullable | string',
            'product_tags_en'      => 'required | string',
            'product_tags_bn'      => 'nullable | string',
            'product_color_en'     => 'required | string',
            'product_color_bn'     => 'nullable | string',
            'product_size_en'      => 'nullable | string',
            'product_size_bn'      => 'nullable | string',
            'product_weight_en'    => 'nullable | string',
            'product_weight_bn'    => 'nullable | string',
            'unit'                 => 'required | string',
            'selling_price'        => 'required | string',
            'discount'             => 'nullable | integer',
            'product_attributes'   => 'nullable | array',
            'discount_type'        => 'required',
            'hightlight_attribute' => 'nullable',
            'product_type'         => 'nullable',
            'hot_deals'            => 'nullable',
            'featured'             => 'nullable',
            'special_offer'        => 'nullable',
            'special_deals'        => 'nullable',
            'short_desc_en'        => 'required',
            'short_desc_bn'        => 'nullable',
            'full_desc_en'         => 'required',
            'full_desc_bn'         => 'nullable',
            'is_price_variant'     => 'boolean',
            'hide_size_weight'     => 'boolean',
            'price_variants'     => 'nullable | array',
        ], [
            'brand_id.required'          => 'Please Select A Brand',
            'category_id.required'       => 'Please Select A Category',
            'subcategory_id.required'    => 'Please Select A Sub Category',
            'subsubcategory_id.required' => 'Please Select A Sub Sub Category',
            'product_name_en.required'   => 'Please Enter Product Name English',
            'product_code.required'      => 'Please Enter Product Code',
            'product_quantity.required'  => 'Please Enter Product Quantity',
            'product_tags_en.required'   => 'Please Enter Product Tags English',
            'product_color_en.required'  => 'Please Enter Product Color English',
            'product_size_en.required'   => 'Please Enter Product Size English',
            'selling_price.required'     => 'Please Enter Product Selling Price',
            'short_desc_en.required'     => 'Enter Product Short Description English',
            'full_desc_en.required'      => 'Enter Product Full Description English',
        ]);
   
        if($request->is_price_variant){
            $request->validate([
                'variants.*.unit' => 'required',
                'variants.*.price' => 'required',
                'variants.*.convert_value' => 'required',
            ]);
        }
        
        try {

            if ($request->discount_type === 'parcent') {
                if (intval($request->discount) > 99) {
                    return redirect()->back()->withError("Discount Should be less than 100");
                }

                $discountAmount = ($request->selling_price * $request->discount) / 100;
                $discountPrice  = round($request->selling_price - $discountAmount);
            }
            else {
                $discountPrice = round($request->selling_price - $request->discount);
            }
            $product = Product::with('productAttributes')->findOrFail($productId);
            $product->update([
                'brand_id'             => $request->brand_id,
                'category_id'          => $request->category_id,
                'subcategory_id'       => $request->subcategory_id,
                'subsubcategory_id'    => $request->subsubcategory_id,
                'product_name_en'      => $request->product_name_en,
                'product_name_bn'      => $request->product_name_bn,
                'product_slug_en'      => Str::slug($request->product_name_en, '-'),
                'product_slug_bn'      => str_replace('', '-', $request->product_name_bn),
                'product_code'         => $request->product_code,
                'product_quantity_en'  => $request->product_quantity_en,
                'product_quantity_bn'  => $request->product_quantity_bn,
                'product_tags_en'      => $request->product_tags_en,
                'product_tags_bn'      => $request->product_tags_bn,
                'product_color_en'     => $request->product_color_en,
                'product_color_bn'     => $request->product_color_bn,
                'product_size_en'      => $request->product_size_en,
                'product_size_bn'      => $request->product_size_bn,
                'product_weight_en'    => $request->product_weight_en,
                'unit'                 => $request->unit,
                'selling_price'        => $request->selling_price,
                'discount'             => $request->discount,
                'discount_type'        => $request->discount_type,
                'discount_price'       => $discountPrice,
                'is_price_variant'     => $request->is_price_variant,
                'price_variants'       => $request->variants,
                'hightlight_attribute' => $request->hightlight_attribute,
                'product_type'         => $request->product_type,
                'hot_deals'            => $request->hot_deals,
                'featured'             => $request->featured,
                'special_offer'        => $request->special_offer,
                'special_deals'        => $request->special_deals,
                'hide_size_weight'     => (int)$request->hide_size_weight,
                'short_desc_en'        => $request->short_desc_en,
                'short_desc_bn'        => $request->short_desc_bn,
                'full_desc_en'         => $request->full_desc_en,
                'full_desc_bn'         => $request->full_desc_bn,
            ]);

            if($request->product_attributes !=null){
                foreach ($product->productAttributes as $attribute)
            {

                ProductAttribute::findOrFail($attribute->id)->delete();
            }
            foreach ($request->product_attributes as $key => $product_attribute)
            {

                ProductAttribute::create([
                    'product_id'      => $productId,
                    'attribute_name'  => $key,
                    'attribute_value' => $product_attribute,
                ]);
            }
            }
            return redirect()->route('products.all')->withSuccess("{$request->product_name_en} Successfully Updated");
        }
        catch (\Exception $e) {
            return redirect()->back()->withInput()->withError($e->getMessage());
        }
    }
    public function editImage($productId)
    {
        $product       = Product::findOrFail($productId);
        $productImages = ProductMultipleImage::where('product_id', $productId)->get();
        return view('admin.product.edit-image', compact('product', 'productImages'));
    }
    public function thumbnailUpdate(Request $request, $productId)
    {
        $oldThumbnail = Product::findOrFail($productId);
        
        if ($request->hasFile('product_thumbnail'))
        {
            if($oldThumbnail?->product_thumbnail){
                unlink($oldThumbnail->product_thumbnail);
            }
            $productThumbnail = $request->file('product_thumbnail');
            $thumbnailName    = uniqid() . time() . '.' . $productThumbnail->extension();
            Image::make($productThumbnail)->resize(917, 1000)->save('uploads/products/thumbnails/' . $thumbnailName);
            $thumbnailUrl = 'uploads/products/thumbnails/' . $thumbnailName;
            Product::findOrFail($productId)->update([
                'product_thumbnail' => $thumbnailUrl,
            ]);
            return redirect()->back()->withSuccess('Product Thumbnail Successfull Updated');
        }
        else
        {
            return redirect()->back()->withInput()->withError('Product Thumbnail Update Fail');
        }
    }
    public function deleteProductImage($productImageId)
    {
        $oldImage = ProductMultipleImage::findOrFail($productImageId);
        if ($oldImage)
        {
            unlink($oldImage->product_image_name);
            ProductMultipleImage::findOrFail($productImageId)->delete();
            return redirect()->back()->withSuccess('Product Image Deleted Successfully');
        }
        else
        {
            return redirect()->back()->withError('Product Image Delete Fail!!');
        }
    }
    public function addProductImages(Request $request, $productId)
    {
        try {

            $product = Product::findOrFail($productId);
            if ($request->hasFile('product_multiple_image'))
            {
                $productImages = $request->file('product_multiple_image');
                foreach ($productImages as $productImage)
                {
                    $imageName = uniqid() . time() . '.' . $productImage->extension();
                    Image::make($productImage)->resize(917, 1000)->save('uploads/products/multiple-images/' . $imageName);
                    $imageUrl = 'uploads/products/multiple-images/' . $imageName;
                    $product->images()->create([
                        'product_image_name' => $imageUrl,
                    ]);
                }
            }
            return redirect()->back()->withSuccess('Product Images Added Successfully');

        }
        catch (\Exception$e)
        {
            return redirect()->back()->withSuccess($e->getMessage());
        }
    }
    
    public function updateImage(Request $request, $id)
    {
         $request->validate([
            'product_image' => 'required | image',
        ], [
            'product_image.required' => 'Please Enter Product Image',
        ]);
        
        if(!$request->hasFile('product_image')){
            return redirect()->back()->withError("Something went wrong");
        }
        try {
            
            $productMultiImage = ProductMultipleImage::findOrFail($id);
            
            $productImage = $request->file('product_image');
            
            $imageName = uniqid() . time() . '.' . $productImage->extension();
            
            unlink($productMultiImage->product_image_name);
            
            Image::make($productImage)->resize(917, 1000)->save('uploads/products/multiple-images/' . $imageName);
            $imageUrl = 'uploads/products/multiple-images/' . $imageName;
            
            $productMultiImage->update([
                'product_image_name' => $imageUrl,
                'product_image_size' => round($productImage->getSize()/1024,2),
                ]);
            
            return redirect()->back()->withSuccess('Image is updated');
        }
        catch (\Exception$e)
        {
            return redirect()->back()->withError($e->getMessage());
        }
        
    }
    public function activeProduct($productId)
    {
        try {
            Product::findOrFail($productId)->update([
                'status' => 1,
            ]);

            return redirect()->back()->withSuccess('This Product is Active Now');
        }
        catch (\Exception$e)
        {
            return redirect()->back()->withError($e->getMessage());
        }

    }
    public function inactiveProduct($productId)
    {
        try {
            Product::findOrFail($productId)->update([
                'status' => 0,
            ]);
            return redirect()->back()->withSuccess('This Product is Inactive Now');
        }
        catch (\Exception$e)
        {
            return redirect()->back()->withError($e->getMessage());
        }

    }
    public function search(Request $request)
    {
        $request->validate([
            'search'=>'required | min:2'
        ]);
        try {
            
            $products = Product::where('product_name_en', 'LIKE', '%' . $request->search . '%')
            ->orWhere('product_name_bn', 'LIKE', '%' . $request->search . '%')
            ->orWhere('product_tags_en', 'LIKE', '%' . $request->search . '%')
            ->orWhere('product_tags_bn', 'LIKE', '%' . $request->search . '%')
            ->orWhere('short_desc_en', 'LIKE', '%' . $request->search . '%')
            ->orWhere('short_desc_bn', 'LIKE', '%' . $request->search . '%')
            ->orWhere('full_desc_en', 'LIKE', '%' . $request->search . '%')
            ->orWhere('full_desc_bn', 'LIKE', '%' . $request->search . '%')->get();
            return view('admin.product.search', compact('products'));
        }
        catch (\Exception$e)
        {
            return redirect()->back()->withError($e->getMessage());
        }
    }
    public function delete($productId)
    {
        $product = Product::findOrFail($productId);
        unlink($product->product_thumbnail);
        $product->delete();
        $productImages = ProductMultipleImage::where('product_id', $productId)->get();
        foreach ($productImages as $productImage)
        {
            $image = ProductMultipleImage::findOrFail($productImage->id);
            unlink($image->product_image_name);
            $image->delete();
        }
        return redirect()->back()->withSuccess('Product Deleted Successfully');

    }
}