<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\AdminRole;
use App\Models\Permission;
use Illuminate\Support\Facades\DB;

class AdminRoleController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if(!\App\Helpers\AdminPermission::isPermitted('view_role')){
            abort(419);
        }
        return view('admin.role.index',['roles' => AdminRole::query()->with(['permissions'])->get()]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        if(!\App\Helpers\AdminPermission::isPermitted('create_role')){
            abort(419);
        }
        return view('admin.role.create',['permissions' => Permission::query()->select('id','name')->get()]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if(!\App\Helpers\AdminPermission::isPermitted('create_role')){
            abort(419);
        }
        $request->validate([
            'name' => ['required','string','min:2'],
            'permissions' => ['required','array','min:1'],
        ]);
        
        DB::beginTransaction();
        
        try {
            
            $role = new AdminRole();
            $role->name = $request->input('name');
            $role->save();
            
            $role->permissions()->attach($request->input('permissions'));
            
            DB::commit();
            
            return redirect()->back()->withSuccess("Role Created Successfully");
            
        }catch(\Exception $e){
            
            DB::rollback();
            
            return redirect()->back()->withError($e->getMessage());
        }
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        if(!\App\Helpers\AdminPermission::isPermitted('edit_role')){
            abort(419);
        }
        $role = AdminRole::query()->with(['permissions'])->findOrFail($id);
        return view('admin.role.edit',[
            'role' => $role,
            'role_permissions' => $role->permissions()->pluck('id')->toArray(),
            'permissions' => Permission::query()->select('id','name')->get(),
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        if(!\App\Helpers\AdminPermission::isPermitted('edit_role')){
            abort(419);
        }
        
        if($id == 1){
            return redirect()->back()->withError("Can not Edit Super Admin Role");
        }
        
        $request->validate([
            'name' => ['required','string','min:2'],
            'permissions' => ['required','array','min:1'],
        ]);
        
        DB::beginTransaction();
        
        try {
            
            $role = AdminRole::query()->findOrFail($id);
            $role->name = $request->input('name');
            $role->update();
            
            $role->permissions()->sync($request->input('permissions'));
            
            DB::commit();
            
            return redirect()->back()->withSuccess("Role Updated Successfully");
            
        }catch(\Exception $e){
            
            DB::rollback();
            
            return redirect()->back()->withError($e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        if(!\App\Helpers\AdminPermission::isPermitted('delete_role')){
            abort(419);
        }
        
        if($id == 1){
            return redirect()->back()->withError("Can not Delete Super Admin Role");
        }
        try {
            
            DB::table('role_permission')->where('admin_role_id',$id)->delete();
            
            $role = AdminRole::query()->findOrFail($id)->delete();
            
            return redirect()->back()->withSuccess("Role Deleted Successfully");
            
        }catch(\Exception $e){
            
            
            return redirect()->back()->withError($e->getMessage());
        }
    }
}
