<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Deal;
use Illuminate\Http\Request;
use Intervention\Image\Facades\Image;

class DealController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return view('admin.deal.index', ['deals' => Deal::all()]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.deal.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'deal_type'  => 'required | string',
            'deal_name'  => 'required | string',
            'start_date' => 'required',
            'end_date'   => 'required',
            'image'      => 'required | file',
            'status'     => 'required | string',
        ]);
        try {
            if ($request->hasFile('image'))
            {
                $image     = $request->file('image');
                $imageName = hexdec(uniqid()) . '.' . $image->getClientOriginalExtension();
                Image::make($image)->resize(166, 110)->save('uploads/deal/' . $imageName);
                $url = 'uploads/deal/' . $imageName;

            }
            Deal::create([
                'deal_type'  => $request->deal_type,
                'deal_name'  => $request->deal_name,
                'start_date' => $request->start_date,
                'end_date'   => $request->end_date,
                'image'      => $url,
                'status'     => $request->status,
            ]);
            return redirect()->route('admin.deals')->withSuccess("{$request->deal_name} Added Successfully");
        }
        catch (\Exception$e)
        {
            return redirect()->back()->withError($e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Deal  $deal
     * @return \Illuminate\Http\Response
     */
    public function show(Deal $deal)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Deal  $deal
     * @return \Illuminate\Http\Response
     */
    public function edit($dealId)
    {
        return view('admin.deal.edit', ['deal' => Deal::findOrfail($dealId)]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Deal  $deal
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $dealId)
    {
        $request->validate([
            'deal_type'  => 'required | string',
            'deal_name'  => 'required | string',
            'start_date' => 'required',
            'end_date'   => 'required',
            'image'      => 'nullable | file',
            'status'     => 'required | string',
        ]);
        try {
            if ($request->hasFile('image'))
            {
                unlink($request->old_image);
                $image     = $request->file('image');
                $imageName = hexdec(uniqid()) . '.' . $image->getClientOriginalExtension();
                Image::make($image)->resize(166, 110)->save('uploads/deal/' . $imageName);
                $url = 'uploads/deal/' . $imageName;

            }
            else
            {
                $url = $request->old_image;
            }
            Deal::findOrFail($dealId)->update([
                'deal_type'  => $request->deal_type,
                'deal_name'  => $request->deal_name,
                'start_date' => $request->start_date,
                'end_date'   => $request->end_date,
                'image'      => $url,
                'status'     => $request->status,
            ]);
            return redirect()->route('admin.deals')->withSuccess("{$request->deal_name} Updated Successfully");
        }
        catch (\Exception$e)
        {
            return redirect()->back()->withError($e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Deal  $deal
     * @return \Illuminate\Http\Response
     */
    public function destroy($dealId)
    {
        try {
            $deal = Deal::findOrFail($dealId);
            unlink($deal->image);
            $deal->delete();
            return redirect()->back()->withSuccess('Deal Deleted Successfully');
        }
        catch (\Exception$e)
        {
            return redirect()->back()->withError($e->getMessage());
        }
    }
}